
//#region ../shared/src/index.ts
function* forEachTouchingNode(ts, sourceFile, position) {
	yield* binaryVisit(ts, sourceFile, sourceFile, position);
}
function* binaryVisit(ts, sourceFile, node, position) {
	const nodes = [];
	ts.forEachChild(node, (child) => {
		nodes.push(child);
	});
	let left = 0;
	let right = nodes.length - 1;
	while (left <= right) {
		const mid = Math.floor((left + right) / 2);
		const node$1 = nodes[mid];
		const start = node$1.getStart(sourceFile);
		const end = node$1.getEnd();
		if (position < start) right = mid - 1;
		else if (position > end) left = mid + 1;
		else {
			yield node$1;
			yield* binaryVisit(ts, sourceFile, node$1, position);
			return;
		}
	}
}

//#endregion
//#region src/index.ts
const plugin = (module$1) => {
	const { typescript: ts } = module$1;
	return { create(info) {
		for (const [key, method] of [
			["findRenameLocations", findRenameLocations.bind(null, ts, info)],
			["findReferences", findReferences.bind(null, ts, info)],
			["getDefinitionAndBoundSpan", getDefinitionAndBoundSpan.bind(null, ts, info)]
		]) {
			const original = info.languageService[key];
			info.languageService[key] = method(original);
		}
		return info.languageService;
	} };
};
var src_default = plugin;
const declarationRE = /\.d\.(?:c|m)?ts$/;
function findRenameLocations(ts, info, findRenameLocations$1) {
	return (...args) => {
		const result = findRenameLocations$1(...args);
		if (!result?.length) return result;
		const program = info.languageService.getProgram();
		const preferences = typeof args[4] === "object" ? args[4] : {};
		const locations = [...result];
		for (const location of result) {
			const sourceFile = program.getSourceFile(location.fileName);
			if (!sourceFile) continue;
			if (!declarationRE.test(location.fileName)) continue;
			const node = visitImports(ts, location.textSpan, sourceFile);
			if (!node) continue;
			const position = node.getStart(sourceFile);
			const res = findRenameLocations$1(location.fileName, position, false, false, preferences);
			if (res?.length) locations.push(...res);
		}
		return locations;
	};
}
function findReferences(ts, info, findReferences$1) {
	return (...args) => {
		const result = findReferences$1(...args);
		if (!result?.length) return result;
		const program = info.languageService.getProgram();
		for (const symbol of result) {
			const references = new Set(symbol.references);
			for (const reference of symbol.references) {
				const sourceFile = program.getSourceFile(reference.fileName);
				if (!sourceFile) continue;
				if (!declarationRE.test(reference.fileName)) continue;
				const node = visitImports(ts, reference.textSpan, sourceFile);
				if (!node) continue;
				const position = node.getStart(sourceFile);
				const res = info.languageService.getReferencesAtPosition(reference.fileName, position)?.filter((entry) => entry.fileName !== reference.fileName || position < entry.textSpan.start || position > entry.textSpan.start + entry.textSpan.length);
				if (res?.length) {
					for (const reference$1 of res) references.add(reference$1);
					references.delete(reference);
				}
			}
			symbol.references = [...references];
		}
		return result;
	};
}
function getDefinitionAndBoundSpan(ts, info, getDefinitionAndBoundSpan$1) {
	return (...args) => {
		const result = getDefinitionAndBoundSpan$1(...args);
		if (!result?.definitions?.length) return result;
		const program = info.languageService.getProgram();
		const definitions = new Set(result.definitions);
		for (const definition of result.definitions) {
			const sourceFile = program.getSourceFile(definition.fileName);
			if (!sourceFile) continue;
			if (!declarationRE.test(definition.fileName)) continue;
			const node = visitImports(ts, definition.textSpan, sourceFile);
			if (!node) continue;
			const position = node.getStart(sourceFile);
			const res = getDefinitionAndBoundSpan$1(definition.fileName, position);
			if (res?.definitions?.length) {
				for (const definition$1 of res.definitions) definitions.add(definition$1);
				definitions.delete(definition);
			}
		}
		return {
			definitions: [...definitions],
			textSpan: result.textSpan
		};
	};
}
function visitImports(ts, textSpan, sourceFile) {
	for (const node of forEachTouchingNode(ts, sourceFile, textSpan.start)) {
		let target;
		if (ts.isPropertySignature(node) && node.type) {
			const args = [
				ts,
				node.name,
				node.type,
				textSpan,
				sourceFile
			];
			target = forwardTypeofImport(...args) ?? backwardTypeofImport(...args);
		} else if (ts.isVariableDeclaration(node) && ts.isIdentifier(node.name) && node.type) {
			const args = [
				ts,
				node.name,
				node.type,
				textSpan,
				sourceFile
			];
			target = forwardTypeofImport(...args) ?? backwardTypeofImport(...args);
		}
		if (target) return target;
	}
}
function forwardTypeofImport(ts, name, type, textSpan, sourceFile) {
	const [start, end] = getStartEnd(name, sourceFile);
	if (start !== textSpan.start || end - start !== textSpan.length) return;
	while (ts.isTypeReferenceNode(type) && type.typeArguments?.length) type = type.typeArguments[0];
	if (ts.isIndexedAccessTypeNode(type)) return type.indexType;
	else if (ts.isImportTypeNode(type)) return type.qualifier ?? type.argument;
}
function backwardTypeofImport(ts, name, type, textSpan, sourceFile) {
	while (ts.isTypeReferenceNode(type) && type.typeArguments?.length) type = type.typeArguments[0];
	let start;
	let end;
	if (ts.isIndexedAccessTypeNode(type)) [start, end] = getStartEnd(type.objectType, sourceFile);
	else if (ts.isImportTypeNode(type)) [start, end] = getStartEnd(type.qualifier ?? type.argument, sourceFile);
	else return;
	if (start === textSpan.start && end - start === textSpan.length) return name;
}
function getStartEnd(node, sourceFile) {
	return [node.getStart(sourceFile), node.getEnd()];
}

//#endregion
module.exports = src_default;